<?php
// track.php - 统一的统计和跟踪处理文件

// 获取请求参数
$appId = $_GET['id'] ?? '';
$action = $_GET['action'] ?? ''; // stats, view 或 download
$platform = $_GET['platform'] ?? 'unknown'; // 平台参数

// 验证应用ID
if (empty($appId)) {
    header('HTTP/1.1 400 Bad Request');
    echo json_encode(['error' => '应用ID不能为空']);
    exit;
}

// 确保stats目录存在
if (!is_dir('stats')) {
    mkdir('stats', 0755, true);
}

$statsFile = "stats/{$appId}.json";

// 处理不同的操作类型
if ($action === 'stats') {
    // 获取统计信息
    handleStatsRequest($appId, $statsFile);
} elseif ($action === 'view' || $action === 'download') {
    // 处理跟踪请求
    handleTrackingRequest($appId, $action, $platform, $statsFile);
} else {
    // 无效的操作类型
    header('HTTP/1.1 400 Bad Request');
    echo json_encode(['error' => '无效的操作类型']);
    exit;
}

/**
 * 处理获取统计信息的请求
 */
function handleStatsRequest($appId, $statsFile) {
    // 加载应用数据
    $appDataFile = "data/{$appId}.json";
    if (!file_exists($appDataFile)) {
        header('HTTP/1.1 404 Not Found');
        echo json_encode(['error' => '应用数据不存在']);
        exit;
    }

    $appData = json_decode(file_get_contents($appDataFile), true);

    if (!file_exists($statsFile)) {
        // 如果统计文件不存在，返回初始数据，但需要包含所有平台
        $platformDownloads = [];
        // 从应用数据中获取所有平台
        if (!empty($appData['downloads'])) {
            foreach ($appData['downloads'] as $download) {
                $platform = $download['platform'];
                $platformDownloads[$platform] = 0;
            }
        }
        $stats = [
            'viewCount' => 0,
            'downloadCount' => 0,
            'platformDownloads' => $platformDownloads
        ];
    } else {
        $stats = json_decode(file_get_contents($statsFile), true);
        // 确保platformDownloads存在
        $stats['platformDownloads'] = $stats['platformDownloads'] ?? [];
        
        // 从应用数据中获取所有平台，补充缺失的平台为0
        if (!empty($appData['downloads'])) {
            foreach ($appData['downloads'] as $download) {
                $platform = $download['platform'];
                if (!isset($stats['platformDownloads'][$platform])) {
                    $stats['platformDownloads'][$platform] = 0;
                }
            }
        }
    }

    header('Content-Type: application/json');
    echo json_encode([
        'viewCount' => $stats['viewCount'] ?? 0,
        'downloadCount' => $stats['downloadCount'] ?? 0,
        'platformDownloads' => $stats['platformDownloads'] ?? []
    ]);
}

/**
 * 处理跟踪请求（查看或下载）
 */
function handleTrackingRequest($appId, $action, $platform, $statsFile) {
    // 初始化统计数据结构
    $stats = [];
    if (file_exists($statsFile)) {
        $stats = json_decode(file_get_contents($statsFile), true);
    } else {
        $stats = [
            'viewCount' => 0,
            'downloadCount' => 0,
            'dailyViews' => [],
            'dailyDownloads' => [],
            'platformDownloads' => [] // 平台下载统计
        ];
    }

    // 当前日期
    $today = date('Y-m-d');

    // 确保数据结构存在
    $stats['dailyViews'] = $stats['dailyViews'] ?? [];
    $stats['dailyDownloads'] = $stats['dailyDownloads'] ?? [];
    $stats['platformDownloads'] = $stats['platformDownloads'] ?? [];

    // 更新统计数据
    if ($action === 'view') {
        $stats['viewCount']++;
        
        if (isset($stats['dailyViews'][$today])) {
            $stats['dailyViews'][$today]++;
        } else {
            $stats['dailyViews'][$today] = 1;
        }
    } elseif ($action === 'download') {
        $stats['downloadCount']++;
        
        // 更新平台下载统计
        if (isset($stats['platformDownloads'][$platform])) {
            $stats['platformDownloads'][$platform]++;
        } else {
            $stats['platformDownloads'][$platform] = 1;
        }
        
        // 更新总每日下载统计
        if (isset($stats['dailyDownloads'][$today])) {
            $stats['dailyDownloads'][$today]++;
        } else {
            $stats['dailyDownloads'][$today] = 1;
        }
    }

    // 保存更新后的数据
    file_put_contents($statsFile, json_encode($stats));

    // 返回成功响应
    header('Content-Type: application/json');
    echo json_encode(['success' => true]);
}
?>